<?php
/**
 * Test Reminder Functionality
 * This script tests if SMS, Email, and WhatsApp reminders are working properly
 * Run this to debug reminder sending issues in credit.php
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../config/db.php';
require_once 'api_helpers.php';
require_once '../includes/functions.php';
require_once '../includes/sms_functions.php';
require_once '../includes/email_functions.php';
require_once '../includes/whatsapp_functions.php';
require_once '../includes/email_templates.php';

?>
<!DOCTYPE html>
<html>
<head>
    <title>Reminder Functionality Test</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .success { background: #c8e6c9; padding: 10px; margin: 10px 0; border-radius: 5px; }
        .error { background: #ffcdd2; padding: 10px; margin: 10px 0; border-radius: 5px; }
        .warning { background: #fff3e0; padding: 10px; margin: 10px 0; border-radius: 5px; }
        .info { background: #e3f2fd; padding: 10px; margin: 10px 0; border-radius: 5px; }
        pre { background: #f5f5f5; padding: 10px; border-radius: 5px; overflow-x: auto; }
    </style>
</head>
<body>

<h1>📧 Reminder Functionality Test</h1>
<p>Testing if SMS, Email, and WhatsApp reminders are working properly in credit.php...</p>

<?php
// Get a test customer with unpaid balance
try {
    $stmt = $pdo->prepare("
        SELECT c.id, c.name, c.phone, c.email,
               SUM(ct.amount - COALESCE(cp_summary.total_paid, 0) - COALESCE(cp_summary.total_waived, 0)) as balance
        FROM customers c
        LEFT JOIN credit_transactions ct ON c.id = ct.customer_id
        LEFT JOIN (
            SELECT 
                credit_transaction_id,
                SUM(CASE WHEN is_waived = 0 OR is_waived IS NULL THEN amount ELSE 0 END) as total_paid,
                SUM(CASE WHEN is_waived = 1 THEN amount ELSE 0 END) as total_waived
            FROM credit_payments
            GROUP BY credit_transaction_id
        ) cp_summary ON ct.id = cp_summary.credit_transaction_id
        WHERE ct.id IS NOT NULL
        GROUP BY c.id, c.name, c.phone, c.email
        HAVING balance > 0.001
        ORDER BY balance DESC
        LIMIT 1
    ");
    $stmt->execute();
    $testCustomer = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$testCustomer) {
        echo "<div class='warning'>";
        echo "❌ <strong>No customers with unpaid balances found.</strong><br>";
        echo "Please ensure there are customers with credit balances to test reminders.";
        echo "</div>";
        exit;
    }

    echo "<h2>🧪 Test Customer Selected:</h2>";
    echo "<ul>";
    echo "<li><strong>Name:</strong> " . htmlspecialchars($testCustomer['name']) . "</li>";
    echo "<li><strong>Phone:</strong> " . htmlspecialchars($testCustomer['phone'] ?: 'Not provided') . "</li>";
    echo "<li><strong>Email:</strong> " . htmlspecialchars($testCustomer['email'] ?: 'Not provided') . "</li>";
    echo "<li><strong>Balance:</strong> Ksh " . number_format($testCustomer['balance'], 2) . "</li>";
    echo "</ul>";

} catch (Exception $e) {
    echo "<div class='error'>";
    echo "❌ <strong>Database Error:</strong> " . htmlspecialchars($e->getMessage());
    echo "</div>";
    exit;
}

// Test message
$testMessage = "Dear " . $testCustomer['name'] . ",\n\n";
$testMessage .= "This is a TEST payment reminder for your outstanding balance of Ksh " . number_format($testCustomer['balance'], 2) . ".\n\n";
$testMessage .= "📧 This is a test message from the POS system.\n";
$testMessage .= "Contact us for payment methods.\n\n";
$testMessage .= "Thank you!\nPOS System";

echo "<h2>📋 Test Message:</h2>";
echo "<pre>" . htmlspecialchars($testMessage) . "</pre>";

echo "<h2>🔧 Configuration Check:</h2>";

// Check SMS Settings
echo "<h3>1. SMS Configuration:</h3>";
try {
    $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'sms_settings'");
    $stmt->execute();
    $smsSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    
    $smsEnabled = ($smsSettings['enable_sms'] ?? '0') === '1';
    $smsConfigured = !empty($smsSettings['sms_api_key']) && !empty($smsSettings['sms_api_secret']) && !empty($smsSettings['sms_sender_id']);
    
    echo "<ul>";
    echo "<li><strong>SMS Enabled:</strong> " . ($smsEnabled ? "✅ Yes" : "❌ No") . "</li>";
    echo "<li><strong>API Key:</strong> " . (!empty($smsSettings['sms_api_key']) ? "✅ Set" : "❌ Not set") . "</li>";
    echo "<li><strong>API Secret:</strong> " . (!empty($smsSettings['sms_api_secret']) ? "✅ Set" : "❌ Not set") . "</li>";
    echo "<li><strong>Sender ID:</strong> " . (!empty($smsSettings['sms_sender_id']) ? "✅ Set (" . htmlspecialchars($smsSettings['sms_sender_id']) . ")" : "❌ Not set") . "</li>";
    echo "<li><strong>Customer has phone:</strong> " . (!empty($testCustomer['phone']) ? "✅ Yes" : "❌ No") . "</li>";
    echo "</ul>";
    
} catch (Exception $e) {
    echo "<div class='error'>❌ Error checking SMS settings: " . htmlspecialchars($e->getMessage()) . "</div>";
}

// Check Email Settings
echo "<h3>2. Email Configuration:</h3>";
try {
    $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'email_settings'");
    $stmt->execute();
    $emailSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    
    $emailEnabled = ($emailSettings['enable_email'] ?? '0') === '1';
    $emailConfigured = !empty($emailSettings['smtp_host']) && !empty($emailSettings['smtp_port']) && 
                      !empty($emailSettings['smtp_username']) && !empty($emailSettings['smtp_password']);
    
    echo "<ul>";
    echo "<li><strong>Email Enabled:</strong> " . ($emailEnabled ? "✅ Yes" : "❌ No") . "</li>";
    echo "<li><strong>SMTP Host:</strong> " . (!empty($emailSettings['smtp_host']) ? "✅ Set (" . htmlspecialchars($emailSettings['smtp_host']) . ")" : "❌ Not set") . "</li>";
    echo "<li><strong>SMTP Port:</strong> " . (!empty($emailSettings['smtp_port']) ? "✅ Set (" . htmlspecialchars($emailSettings['smtp_port']) . ")" : "❌ Not set") . "</li>";
    echo "<li><strong>SMTP Username:</strong> " . (!empty($emailSettings['smtp_username']) ? "✅ Set (" . htmlspecialchars($emailSettings['smtp_username']) . ")" : "❌ Not set") . "</li>";
    echo "<li><strong>SMTP Password:</strong> " . (!empty($emailSettings['smtp_password']) ? "✅ Set" : "❌ Not set") . "</li>";
    echo "<li><strong>Encryption:</strong> " . htmlspecialchars($emailSettings['smtp_encryption'] ?? 'Not set') . "</li>";
    echo "<li><strong>Customer has email:</strong> " . (!empty($testCustomer['email']) ? "✅ Yes" : "❌ No") . "</li>";
    echo "</ul>";
    
} catch (Exception $e) {
    echo "<div class='error'>❌ Error checking email settings: " . htmlspecialchars($e->getMessage()) . "</div>";
}

// Check WhatsApp Settings
echo "<h3>3. WhatsApp Configuration:</h3>";
try {
    $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'whatsapp_settings'");
    $stmt->execute();
    $whatsappSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    
    $whatsappEnabled = ($whatsappSettings['enable_whatsapp'] ?? '0') === '1';
    
    echo "<ul>";
    echo "<li><strong>WhatsApp Enabled:</strong> " . ($whatsappEnabled ? "✅ Yes" : "❌ No") . "</li>";
    echo "<li><strong>Customer has phone:</strong> " . (!empty($testCustomer['phone']) ? "✅ Yes" : "❌ No") . "</li>";
    echo "</ul>";
    
} catch (Exception $e) {
    echo "<div class='error'>❌ Error checking WhatsApp settings: " . htmlspecialchars($e->getMessage()) . "</div>";
}

echo "<h2>🧪 Actual Function Tests:</h2>";

// Test SMS Function
echo "<h3>1. Testing SMS Function:</h3>";
if (!empty($testCustomer['phone']) && $smsEnabled && $smsConfigured) {
    try {
        echo "<div class='info'>";
        echo "📱 Attempting to send SMS to: " . htmlspecialchars($testCustomer['phone']) . "<br>";
        
        $smsResult = sendSMS($testCustomer['phone'], $testMessage);
        
        if ($smsResult) {
            echo "</div><div class='success'>✅ <strong>SMS sent successfully!</strong></div>";
        } else {
            echo "</div><div class='error'>❌ <strong>SMS failed to send</strong></div>";
        }
        
    } catch (Exception $e) {
        echo "</div><div class='error'>❌ <strong>SMS Error:</strong> " . htmlspecialchars($e->getMessage()) . "</div>";
    }
} else {
    echo "<div class='warning'>";
    echo "⚠️ <strong>SMS Test Skipped:</strong> ";
    if (empty($testCustomer['phone'])) echo "Customer has no phone number. ";
    if (!$smsEnabled) echo "SMS not enabled. ";
    if (!$smsConfigured) echo "SMS not properly configured. ";
    echo "</div>";
}

// Test Email Function
echo "<h3>2. Testing Email Function:</h3>";
if (!empty($testCustomer['email']) && $emailEnabled && $emailConfigured) {
    try {
        echo "<div class='info'>";
        echo "📧 Attempting to send Email to: " . htmlspecialchars($testCustomer['email']) . "<br>";
        
        $emailResult = sendEmail($testCustomer['email'], 'Test Payment Reminder', $testMessage);
        
        if ($emailResult) {
            echo "</div><div class='success'>✅ <strong>Email sent successfully!</strong></div>";
        } else {
            echo "</div><div class='error'>❌ <strong>Email failed to send</strong></div>";
        }
        
    } catch (Exception $e) {
        echo "</div><div class='error'>❌ <strong>Email Error:</strong> " . htmlspecialchars($e->getMessage()) . "</div>";
    }
} else {
    echo "<div class='warning'>";
    echo "⚠️ <strong>Email Test Skipped:</strong> ";
    if (empty($testCustomer['email'])) echo "Customer has no email address. ";
    if (!$emailEnabled) echo "Email not enabled. ";
    if (!$emailConfigured) echo "Email not properly configured. ";
    echo "</div>";
}

// Test WhatsApp Function
echo "<h3>3. Testing WhatsApp Function:</h3>";
if (!empty($testCustomer['phone']) && $whatsappEnabled) {
    try {
        echo "<div class='info'>";
        echo "💬 Attempting to generate WhatsApp link for: " . htmlspecialchars($testCustomer['phone']) . "<br>";
        
        $whatsappResult = sendWhatsApp($testCustomer['phone'], $testMessage);
        
        if ($whatsappResult['success']) {
            echo "</div><div class='success'>";
            echo "✅ <strong>WhatsApp link generated successfully!</strong><br>";
            if (isset($whatsappResult['whatsapp_url'])) {
                echo "🔗 <strong>WhatsApp URL:</strong> <a href='" . htmlspecialchars($whatsappResult['whatsapp_url']) . "' target='_blank'>Click to open WhatsApp</a>";
            }
            echo "</div>";
        } else {
            echo "</div><div class='error'>❌ <strong>WhatsApp failed:</strong> " . htmlspecialchars($whatsappResult['message'] ?? 'Unknown error') . "</div>";
        }
        
    } catch (Exception $e) {
        echo "</div><div class='error'>❌ <strong>WhatsApp Error:</strong> " . htmlspecialchars($e->getMessage()) . "</div>";
    }
} else {
    echo "<div class='warning'>";
    echo "⚠️ <strong>WhatsApp Test Skipped:</strong> ";
    if (empty($testCustomer['phone'])) echo "Customer has no phone number. ";
    if (!$whatsappEnabled) echo "WhatsApp not enabled. ";
    echo "</div>";
}

echo "<h2>📊 Summary & Recommendations:</h2>";

echo "<div class='success'>";
echo "<h3>✅ What's Working:</h3>";
echo "<ul>";

if ($smsEnabled && $smsConfigured && !empty($testCustomer['phone'])) {
    echo "<li>SMS configuration appears complete</li>";
}
if ($emailEnabled && $emailConfigured && !empty($testCustomer['email'])) {
    echo "<li>Email configuration appears complete</li>";
}
if ($whatsappEnabled && !empty($testCustomer['phone'])) {
    echo "<li>WhatsApp configuration appears complete</li>";
}

echo "</ul>";
echo "</div>";

echo "<div class='warning'>";
echo "<h3>⚠️ Issues Found:</h3>";
echo "<ul>";

if (!$smsEnabled) {
    echo "<li><strong>SMS not enabled:</strong> Go to Settings → SMS Settings and enable SMS notifications</li>";
}
if ($smsEnabled && !$smsConfigured) {
    echo "<li><strong>SMS not configured:</strong> Configure API key, secret, and sender ID in SMS settings</li>";
}
if (!$emailEnabled) {
    echo "<li><strong>Email not enabled:</strong> Go to Settings → Email Settings and enable email notifications</li>";
}
if ($emailEnabled && !$emailConfigured) {
    echo "<li><strong>Email not configured:</strong> Configure SMTP settings in Email settings</li>";
}
if (!$whatsappEnabled) {
    echo "<li><strong>WhatsApp not enabled:</strong> Go to Settings → WhatsApp Settings and enable WhatsApp notifications</li>";
}
if (empty($testCustomer['phone'])) {
    echo "<li><strong>Customer missing phone:</strong> SMS and WhatsApp require customer phone numbers</li>";
}
if (empty($testCustomer['email'])) {
    echo "<li><strong>Customer missing email:</strong> Email reminders require customer email addresses</li>";
}

echo "</ul>";
echo "</div>";

echo "<div class='info'>";
echo "<h3>🔧 Next Steps to Fix Issues:</h3>";
echo "<ol>";
echo "<li><strong>Fix Configuration Issues:</strong> Address any configuration problems identified above</li>";
echo "<li><strong>Test in Credit.php:</strong> Go to Credit page and try sending a reminder manually</li>";
echo "<li><strong>Check Browser Console:</strong> Look for JavaScript errors when clicking Send Reminder</li>";
echo "<li><strong>Check Network Tab:</strong> Verify that AJAX calls to send_reminder.php are being made</li>";
echo "<li><strong>Verify Delivery:</strong> Check if SMS/Email/WhatsApp messages are actually received</li>";
echo "</ol>";
echo "</div>";

echo "<hr>";
echo "<p><em>Test completed at " . date('Y-m-d H:i:s') . "</em></p>";
?>

</body>
</html> 